
#include <linux/kernel.h>	/* printk */

#include "iprq.h"
#include "iprq_common.h"


static inline int __get_buddy_index(int index, int shift)
{
	int base, size, buddy;

#ifdef IPRQ_ASSERT
	if (index < 0 || index >= MEMBLK_ALL_NUM) {
		IPRQ_ABORT();
	}
	if (shift > MEMBLK_MAX_SHIFT || shift < MEMBLK_MIN_SHIFT) {
		IPRQ_ABORT();
	}
#endif
	size = 1 << (shift - MEMBLK_MIN_SHIFT);
	base = index & ~(size - 1);
	buddy = base ^ size;

	return buddy;
}

static inline int __memblk_set_block_size_area(struct iprq_memblk_t *memblk, int index, int shift)
{
	int start, i, size;

#ifdef IPRQ_ASSERT
	if (index < 0 || index >= MEMBLK_ALL_NUM) {
		IPRQ_ABORT();
	}
	if (shift > MEMBLK_MAX_SHIFT || shift < MEMBLK_MIN_SHIFT) {
		IPRQ_ABORT();
	}
#endif
	size = 1 << (shift - MEMBLK_MIN_SHIFT);
	start = index & ~(size - 1);

	for (i = start; i < start + size; i++) {
		__memblk_set_block_size(memblk, i, shift);
	}
	return 0;
}

static int __memblk_split_buddy_block(struct iprq_memblk_t *memblk, int index, int newshift)
{
	int shift, buddy;

	for (shift = __memblk_get_block_size(memblk, index) - 1; shift >= newshift; shift--) {
		buddy = __get_buddy_index(index, shift);

		__memblk_set_block_size_area(memblk, buddy, shift);

		memblk->unused_num[shift + 1]--;
		memblk->unused_num[shift] += 2;
#ifdef IPRQ_ASSERT
		if ((int)memblk->unused_num[shift + 1] < 0) {
			IPRQ_ABORT();
		}
#endif
	}

	__memblk_set_block_size_area(memblk, index, newshift);

	return 0;
}

static int __memblk_combine_buddy_block(struct iprq_memblk_t *memblk, int index)
{
	int shift, buddy, master;
	int idx = index;

	while (1) {
		shift = __memblk_get_block_size(memblk, idx);
		buddy = __get_buddy_index(idx, shift);

		if (buddy < 0 || buddy >= MEMBLK_ALL_NUM ||
		    __memblk_get_block_size(memblk, buddy) != shift) {
			break;
		}

		if (__is_memblk_used(memblk, idx) || __is_memblk_used(memblk, buddy)) {
			break;
		}
		master = (idx < buddy) ? idx : buddy;

		__memblk_set_block_size_area(memblk, master, shift + 1);

		memblk->unused_num[shift] -= 2;
		memblk->unused_num[shift + 1]++;
#ifdef IPRQ_ASSERT
		if ((int)memblk->unused_num[shift] < 0) {
			IPRQ_ABORT();
		}
#endif
		idx = master;
	}
	return 0;
}

static inline int __memblk_require_block_size(unsigned int size)
{
	int shift;

	if (size == 0) {
		return -1;
	}
	for (shift = MEMBLK_MIN_SHIFT; shift <= MEMBLK_MAX_SHIFT; shift++) {
		if (!((size - 1) >> shift)) {
			break;
		}
	}
	if (shift > MEMBLK_MAX_SHIFT) {
		return -1;
	}
	return shift;
}

static int __memblk_search_unused_block(struct iprq_memblk_t *memblk, int shift)
{
	int s, i, tmp;

	for (s = shift; s <= MEMBLK_MAX_SHIFT; s++) {
		if (memblk->unused_num[s] > 0) {
			break;
		}
	}
	if (s > MEMBLK_MAX_SHIFT) {
		return -1;
	}
	for (i = 0; i < MEMBLK_ALL_NUM;  ) {
		tmp = __memblk_get_block_size(memblk, i);
#ifdef IPRQ_ASSERT
		if (tmp < MEMBLK_MIN_SHIFT || tmp > MEMBLK_MAX_SHIFT) {
			IPRQ_ABORT();
		}
#endif
		if ((tmp == s) && !__is_memblk_used(memblk, i)) {
			break;
		}
		tmp = (shift > tmp) ? shift : tmp;
		i += (1 << (tmp - MEMBLK_MIN_SHIFT));
	}
#ifdef IPRQ_ASSERT
	if (i >= MEMBLK_ALL_NUM) {
		IPRQ_ABORT();
	}
#endif
	return i;
}

static inline void __memblk_set_used_range(struct iprq_memblk_t *memblk,
					   int start, int len)
{
	int i;
	for (i = start; i < (start + len); i++) {
		__memblk_set_used(memblk, i);
	}
}

static inline void __memblk_clear_used_range(struct iprq_memblk_t *memblk,
					     int start, int len)
{
	int i;
	for (i = start; i < (start + len); i++) {
		__memblk_clear_used(memblk, i);
	}
}

static int __memblk_alloc_block(struct iprq_memblk_t *memblk, unsigned int size)
{
	int shift, index;

	shift = __memblk_require_block_size(size);
	if (shift < 0) {
		return E_IPRQ_INVAL;
	}
	index = __memblk_search_unused_block(memblk, shift);
	if (index < 0) {
		return E_IPRQ_NOMEM;
	}

	if (__memblk_get_block_size(memblk, index) > shift) {
		__memblk_split_buddy_block(memblk, index, shift);
	}

	__memblk_set_used_range(memblk, index, 1 << (shift - MEMBLK_MIN_SHIFT));
	memblk->unused_num[shift]--;
#ifdef IPRQ_ASSERT
	if ((int)memblk->unused_num[shift] < 0) {
		IPRQ_ABORT();
	}
#endif
	return index;
}

static int __memblk_free_block(struct iprq_memblk_t *memblk, int index)
{
	int shift;
	if (index < 0 || index >= MEMBLK_ALL_NUM) {
		return E_IPRQ_INVAL;
	}
	if (__is_memblk_reserved(memblk, index) ||
	    !__is_memblk_used(memblk, index)) {
		return E_IPRQ_BUSY;
	}

	shift = __memblk_get_block_size(memblk, index);

	__memblk_clear_used_range(memblk, index, 1 << (shift - MEMBLK_MIN_SHIFT));
	memblk->unused_num[shift]++;

	__memblk_combine_buddy_block(memblk, index);

	return 0;
}

int iprqX_alloc_mem(unsigned int size)
{
	struct iprq_memblk_t *memblk = iprq_memblk;
	unsigned long flags;
	int index, memid;
#ifdef IPRQ_WARNING
	static unsigned int alloc_err_size = 0;
	int print_flag = 0;
#endif

	if (!is_iprq_initialized()) {
		IPRQ_ABORT();
	}

	flags = LOCK_CLI_SAVE(&memblk->lock);

	index = __memblk_alloc_block(memblk, size);
	if (index < 0) {
		memid = index;
	} else {
		memid =	iprq_index_to_memid(index);
	}

#ifdef IPRQ_WARNING
	if (memid < 0) {
		if (alloc_err_size != size) {
			print_flag = 1;
			alloc_err_size = size;
		}
	} else {
		alloc_err_size = 0;
	}
#endif
	UNLOCK_IRQ_RESTORE(&memblk->lock, flags);

#ifdef IPRQ_WARNING
	if (print_flag) {
		printk(KERN_WARNING
		       "IPRQ Warning: can't allocate memory "
		       "(requested %d bytes)\n", size);
	}
#endif

	return memid;
}

int iprqX_free_mem(int memid)
{
	struct iprq_memblk_t *memblk = iprq_memblk;
	int ret, index;
	unsigned long flags;

	if (!is_iprq_initialized()) {
		IPRQ_ABORT();
	}

	flags = LOCK_CLI_SAVE(&memblk->lock);

	index = iprq_memid_to_index(memid);
	ret = __memblk_free_block(memblk, index);

	UNLOCK_IRQ_RESTORE(&memblk->lock, flags);

	return ret;
}

void *iprqX_attach_mem(int memid)
{
	struct iprq_memblk_t *memblk = iprq_memblk;
	unsigned long flags;
	int index;
	void* ret = NULL;

	if (!is_iprq_initialized()) {
		IPRQ_ABORT();
	}

	flags = LOCK_CLI_SAVE(&memblk->lock);

	index = iprq_memid_to_index(memid);
	if (index < 0 || index >= MEMBLK_ALL_NUM) {
		goto out;
	}

	if (!__is_memblk_used(memblk, index)) {
		goto out;
	}
	ret = (void*)iprq_memid_to_addr(memblk, memid);
out:
	UNLOCK_IRQ_RESTORE(&memblk->lock, flags);

	return ret;
}

int iprqX_detach_mem(void *ptr)
{
	if (!is_iprq_initialized()) {
		IPRQ_ABORT();
	}
	if (((unsigned long)ptr < IPRQ_MEM_ADDR) ||
	    ((unsigned long)ptr >= IPRQ_MEM_ADDR + IPRQ_MEM_SIZE)) {
		return E_IPRQ_INVAL;
	}
	return 0;
}
